﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using DocumentFormat.OpenXml;
using DocumentFormat.OpenXml.Packaging;
using DocumentFormat.OpenXml.Wordprocessing;

namespace DocxFormatterApp
{
    public enum ContributionType { SomeoneWrote, KrodhaReplied, Quote }

    public class Contribution
    {
        public ContributionType Type { get; set; }
        public string Text { get; set; } = "";
        public int FontSize { get; set; } // half-points
        public int ParagraphIndex { get; set; }
    }

    public class ConversationBlock
    {
        public string TopicTitle { get; set; } = "";
        public string TopicLink { get; set; } = "";
        public List<Contribution> Contributions { get; set; } = new List<Contribution>();
    }

    class Program
    {
        // Hex codes for green text (for title detection).
        private static readonly HashSet<string> GreenHexCodes = new HashSet<string>
        {
            "008000", "00aa00", "00ff00", "008b00", "00b050"
        };

        static void Main(string[] args)
        {
            // Check for running WINWORD processes.
            var wordProcesses = Process.GetProcessesByName("WINWORD");
            if (wordProcesses.Length > 0)
            {
                Console.WriteLine("Warning: Microsoft Word appears to be running. Please save your work and close Word before proceeding.");
                Console.Write("Force close Word now? (Y/N): ");
                string response = Console.ReadLine();
                if (response.Trim().ToLower() == "y")
                {
                    foreach (Process proc in wordProcesses)
                    {
                        try { proc.Kill(); }
                        catch (Exception ex) { Console.WriteLine("Could not close process: " + ex.Message); }
                    }
                    Console.WriteLine("Closed all WINWORD processes.");
                }
                else
                {
                    Console.WriteLine("Please close Word manually and then press any key to continue.");
                    Console.ReadKey();
                }
            }

            try
            {
                Console.Write("Enter the path of the original DOCX file: ");
                string inputPath = Console.ReadLine();
                if (!File.Exists(inputPath))
                {
                    Console.WriteLine("Input file does not exist.");
                    return;
                }

                Console.Write("Enter the output DOCX file path: ");
                string outputPath = Console.ReadLine();

                Console.Write("Enter the username for replies (default is 'Krodha'): ");
                string replyUsername = Console.ReadLine();
                if (string.IsNullOrWhiteSpace(replyUsername))
                    replyUsername = "Krodha";

                Console.WriteLine("The document's entries are currently from newest (top) to oldest (bottom).");
                Console.WriteLine("Would you like to keep that order (D for descending),");
                Console.WriteLine("or reverse it so oldest is at the top (A for ascending)?");
                Console.Write("Enter 'A' (ascending) or 'D' (descending): ");
                string orderChoice = Console.ReadLine().Trim().ToUpper();

                // Pass 0: determine dynamic quote size (larger of the two most common sizes).
                var quoteSize = DetectQuoteFontSize(inputPath);
                Console.WriteLine($"DEBUG: Detected quote (large) font size = {quoteSize} (half-points).");

                // Process with this dynamic size.
                List<ConversationBlock> blocks = ProcessDocument(inputPath, quoteSize);
                Console.WriteLine($"DEBUG: ProcessDocument returned {blocks.Count} block(s).");

                if (orderChoice == "A")
                {
                    blocks.Reverse();
                    Console.WriteLine("Reversed blocks so oldest is first (ascending order).");
                }
                else
                {
                    Console.WriteLine("Keeping newest first (descending order).");
                }

                string formattedContent = BuildFormattedContent(blocks, replyUsername);
                Console.WriteLine("DEBUG: Formatted Content:");
                Console.WriteLine(formattedContent);

                CreateOutputDocx(outputPath, blocks, formattedContent);
                Console.WriteLine("Document formatting complete. Check the output file.");
            }
            catch (Exception ex)
            {
                Console.WriteLine("An error occurred: " + ex.Message);
            }

            Console.WriteLine("Press any key to exit...");
            Console.ReadKey();
        }

        // ===== PASS 0: scan font sizes and pick the larger dominant as “quote” size =====
        private static int DetectQuoteFontSize(string inputPath)
        {
            // Fallback in case we cannot detect: treat >= 26 as large (13pt).
            const int fallback = 26;

            var freq = new Dictionary<int, int>(); // size -> count

            using (var wordDoc = WordprocessingDocument.Open(inputPath, false))
            {
                var mainPart = wordDoc.MainDocumentPart;
                var paragraphs = mainPart.Document.Descendants<Paragraph>();

                foreach (var para in paragraphs)
                {
                    string text = para.InnerText;
                    if (string.IsNullOrWhiteSpace(text)) continue;

                    // Titles or separators are ignored for size histogram.
                    var (url, _) = GetFirstHyperlinkWithText(para, mainPart);
                    bool isTitleish = IsGreenParagraph(para, wordDoc) || !string.IsNullOrEmpty(url);
                    bool isSeparator = text.TrimStart().StartsWith("Done!", StringComparison.OrdinalIgnoreCase);
                    if (isTitleish || isSeparator) continue;

                    int size = GetDominantFontSize(para, wordDoc); // half-points
                    if (size <= 0) continue;

                    if (!freq.ContainsKey(size)) freq[size] = 0;
                    freq[size]++;
                }
            }

            if (freq.Count == 0) return fallback;

            // Find two most common distinct sizes.
            var top = freq.OrderByDescending(kv => kv.Value).Select(kv => kv.Key).ToList();
            if (top.Count == 1) return top[0];

            // Pick the larger of the top two as the “quote” size.
            int s1 = top[0];
            int s2 = top[1];
            return Math.Max(s1, s2);
        }

        /// <summary>
        /// Process the DOCX into conversation blocks using a dynamic largeSize (half-points).
        /// </summary>
        static List<ConversationBlock> ProcessDocument(string inputPath, int dynamicLargeSize)
        {
            List<ConversationBlock> blocks = new List<ConversationBlock>();
            ConversationBlock currentBlock = null;
            int paraCount = 0;

            using (WordprocessingDocument wordDoc = WordprocessingDocument.Open(inputPath, false))
            {
                var mainPart = wordDoc.MainDocumentPart;
                var paragraphs = mainPart.Document.Descendants<Paragraph>();
                Console.WriteLine($"DEBUG: Found {paragraphs.Count()} paragraphs in the document.");

                foreach (Paragraph para in paragraphs)
                {
                    paraCount++;
                    string text = para.InnerText.Trim();
                    if (string.IsNullOrEmpty(text)) continue;

                    int fontSize = GetDominantFontSize(para, wordDoc);
                    bool isLarge = fontSize >= dynamicLargeSize; // adaptive
                    bool isGreen = IsGreenParagraph(para, wordDoc);
                    bool isQuoteStyle = IsQuoteStyled(para, wordDoc);
                    bool looksQuote = isLarge || isQuoteStyle;

                    var (url, linkText) = GetFirstHyperlinkWithText(para, mainPart);
                    bool hasLink = !string.IsNullOrEmpty(url);

                    Console.WriteLine($"DEBUG: P{paraCount}: '{text}' (Size: {fontSize}, IsLarge(adapt): {isLarge}, QuoteStyle: {isQuoteStyle}, IsGreen: {isGreen}, HasLink: {hasLink})");

                    // Separator always closes current block
                    if (text.TrimStart().StartsWith("Done!", StringComparison.OrdinalIgnoreCase))
                    {
                        if (currentBlock != null)
                        {
                            blocks.Add(currentBlock);
                            Console.WriteLine("DEBUG: Separator 'Done!' encountered. Closed current block.");
                        }
                        currentBlock = null;
                        continue;
                    }

                    // Title detection
                    bool isTitle = isGreen || hasLink;
                    if (isTitle)
                    {
                        if (currentBlock != null)
                        {
                            blocks.Add(currentBlock);
                            Console.WriteLine($"DEBUG: Completed block. New title trigger => '{text}'");
                        }

                        string titleToUse = !string.IsNullOrWhiteSpace(linkText) ? linkText.Trim() : text;
                        currentBlock = new ConversationBlock
                        {
                            TopicTitle = titleToUse,
                            TopicLink = url ?? ""
                        };

                        Console.WriteLine($"DEBUG: Started new block. Title='{currentBlock.TopicTitle}', Link='{currentBlock.TopicLink}'");
                        continue; // titles don’t become content
                    }

                    // Not a title and no current block => skip until a title appears
                    if (currentBlock == null)
                    {
                        Console.WriteLine($"DEBUG: Skipping paragraph {paraCount} because no block is active (no title encountered yet).");
                        continue;
                    }

                    var contrib = new Contribution
                    {
                        Text = text,
                        FontSize = fontSize,
                        ParagraphIndex = paraCount
                    };

                    if (currentBlock.Contributions.Count == 0)
                    {
                        contrib.Type = looksQuote ? ContributionType.SomeoneWrote : ContributionType.KrodhaReplied;
                    }
                    else
                    {
                        contrib.Type = looksQuote ? ContributionType.Quote : ContributionType.KrodhaReplied;
                    }

                    currentBlock.Contributions.Add(contrib);
                }
            }

            if (currentBlock != null)
            {
                blocks.Add(currentBlock);
                Console.WriteLine("DEBUG: Added final block.");
            }

            return blocks;
        }

        // ————— Helpers —————

        private static (string url, string linkText) GetFirstHyperlinkWithText(Paragraph para, MainDocumentPart mainPart)
        {
            var hyperlink = para.Descendants<DocumentFormat.OpenXml.Wordprocessing.Hyperlink>().FirstOrDefault();
            if (hyperlink != null)
            {
                string rId = hyperlink.Id;
                if (!string.IsNullOrEmpty(rId))
                {
                    var rel = mainPart.HyperlinkRelationships.FirstOrDefault(h => h.Id == rId);
                    if (rel != null)
                        return (rel.Uri.OriginalString, "");
                }
            }

            var instrRun = para.Descendants<Run>()
                .FirstOrDefault(r => r.Descendants<FieldCode>()
                    .Any(fc => fc.Text.Trim().StartsWith("HYPERLINK ", StringComparison.OrdinalIgnoreCase)));

            if (instrRun != null)
            {
                var fieldCodeElem = instrRun.Descendants<FieldCode>().FirstOrDefault();
                if (fieldCodeElem != null)
                {
                    string fieldCode = fieldCodeElem.Text.Trim();
                    var re = new Regex(@"HYPERLINK\s+""([^""]+)""(\s+\\h(.*))?", RegexOptions.IgnoreCase);
                    var match = re.Match(fieldCode);
                    if (match.Success)
                    {
                        string url = match.Groups[1].Value;
                        string linkText = match.Groups[3].Value.Trim();
                        return (url, linkText);
                    }
                }
            }

            return ("", "");
        }

        static bool IsGreenParagraph(Paragraph para, WordprocessingDocument doc)
        {
            foreach (var run in para.Descendants<Run>())
            {
                var rp = run.RunProperties;
                if (rp?.Color != null)
                {
                    var c = rp.Color;
                    if (!string.IsNullOrEmpty(c.Val) &&
                        GreenHexCodes.Contains(c.Val.Value.Trim().ToLower()))
                        return true;

                    if (c.ThemeColor != null)
                    {
                        var theme = c.ThemeColor.Value.ToString();
                        if (theme.Equals("Accent3", StringComparison.OrdinalIgnoreCase) ||
                            theme.Equals("Accent4", StringComparison.OrdinalIgnoreCase))
                            return true;
                    }
                }

                var pmrp = para.ParagraphProperties?.ParagraphMarkRunProperties;
                if (pmrp != null)
                {
                    var colorElem = pmrp.Elements<Color>().FirstOrDefault();
                    if (colorElem != null)
                    {
                        if (!string.IsNullOrEmpty(colorElem.Val) &&
                            GreenHexCodes.Contains(colorElem.Val.Value.Trim().ToLower()))
                            return true;

                        if (colorElem.ThemeColor != null)
                        {
                            var theme = colorElem.ThemeColor.Value.ToString();
                            if (theme.Equals("Accent3", StringComparison.OrdinalIgnoreCase) ||
                                theme.Equals("Accent4", StringComparison.OrdinalIgnoreCase))
                                return true;
                        }
                    }
                }
            }

            // Style-name heuristic as fallback
            var styleId = para.ParagraphProperties?.ParagraphStyleId?.Val?.Value;
            if (!string.IsNullOrEmpty(styleId))
            {
                var stylesPart = doc.MainDocumentPart.StyleDefinitionsPart;
                var style = stylesPart?.Styles?.Elements<Style>()?.FirstOrDefault(s => s.StyleId == styleId);
                var name = style?.StyleName?.Val?.Value?.ToLower() ?? "";
                if (name.Contains("green") || name.Contains("title") || name.Contains("heading"))
                    return true;
            }

            return false;
        }

        // NEW: Quote style detector (Word often uses “Quote” / “Intense Quote” styles)
        private static bool IsQuoteStyled(Paragraph para, WordprocessingDocument doc)
        {
            var styleId = para.ParagraphProperties?.ParagraphStyleId?.Val?.Value;
            if (string.IsNullOrEmpty(styleId)) return false;

            var stylesPart = doc.MainDocumentPart.StyleDefinitionsPart;
            var style = stylesPart?.Styles?.Elements<Style>()?.FirstOrDefault(s => s.StyleId == styleId);
            var name = style?.StyleName?.Val?.Value?.ToLower() ?? "";

            return name.Contains("quote");
        }

        // NEW: dominant (max) font size in half-points across runs; fallback to style; fallback default 24
        private static int GetDominantFontSize(Paragraph para, WordprocessingDocument wordDoc)
        {
            int maxRunSize = 0;

            foreach (Run run in para.Descendants<Run>())
            {
                var rp = run.RunProperties;
                if (rp == null) continue;

                // Check FontSize and ComplexScript size
                int size = ExtractSizeHalfPoints(rp.FontSize?.Val);
                int csSize = ExtractSizeHalfPoints(rp.FontSizeComplexScript?.Val);

                int candidate = Math.Max(size, csSize);
                if (candidate > maxRunSize) maxRunSize = candidate;
            }

            if (maxRunSize > 0) return maxRunSize;

            // Try style-level font size
            var pPr = para.ParagraphProperties;
            if (pPr?.ParagraphStyleId != null)
            {
                string styleId = pPr.ParagraphStyleId.Val.Value;
                var stylesPart = wordDoc.MainDocumentPart.StyleDefinitionsPart;
                var style = stylesPart?.Styles?.Elements<Style>()?.FirstOrDefault(s => s.StyleId == styleId);
                if (style?.StyleRunProperties != null)
                {
                    int s1 = ExtractSizeHalfPoints(style.StyleRunProperties.FontSize?.Val);
                    int s2 = ExtractSizeHalfPoints(style.StyleRunProperties.FontSizeComplexScript?.Val);
                    int candidate = Math.Max(s1, s2);
                    if (candidate > 0) return candidate;
                }
            }

            // Default (12pt = 24 half-points)
            return 24;
        }

        private static int ExtractSizeHalfPoints(StringValue val)
        {
            if (val == null) return 0;
            return int.TryParse(val.Value, out int hp) ? hp : 0;
        }

        static List<Contribution> MergeContributions(List<Contribution> contributions)
        {
            if (contributions.Count == 0) return new List<Contribution>();

            List<Contribution> merged = new List<Contribution>();
            Contribution current = contributions[0];
            merged.Add(current);

            for (int i = 1; i < contributions.Count; i++)
            {
                var next = contributions[i];

                bool sameTypeAndConsecutive =
                    (current.Type == next.Type) &&
                    (next.ParagraphIndex == current.ParagraphIndex + 1) &&
                    (current.Type == ContributionType.KrodhaReplied || current.Type == ContributionType.Quote);

                if (sameTypeAndConsecutive)
                {
                    current.Text += Environment.NewLine + next.Text;
                    current.ParagraphIndex = next.ParagraphIndex;
                }
                else
                {
                    current = next;
                    merged.Add(current);
                }
            }

            return merged;
        }
        static string BuildFormattedContent(List<ConversationBlock> blocks, string replyUsername)
        {
            var sb = new System.Text.StringBuilder();

            foreach (var block in blocks)
            {
                sb.AppendLine("Topic Title:  " + block.TopicTitle);
                sb.AppendLine();

                var mergedContribs = MergeContributions(block.Contributions);

                bool seenAnyQuote = mergedContribs.Any(c => c.Type == ContributionType.SomeoneWrote || c.Type == ContributionType.Quote);
                bool printedFirstNonQuote = false;
                bool printedAnyQuote = false;

                foreach (var contrib in mergedContribs)
                {
                    bool isQuote = contrib.Type == ContributionType.SomeoneWrote || contrib.Type == ContributionType.Quote;

                    if (isQuote)
                    {
                        if (!printedAnyQuote)
                        {
                            sb.AppendLine("Someone wrote:");
                        }
                        else
                        {
                            sb.AppendLine("Quote:");
                        }
                        sb.AppendLine(contrib.Text);
                        sb.AppendLine();
                        printedAnyQuote = true;
                    }
                    else
                    {
                        if (!printedFirstNonQuote)
                        {
                            // If earlier quote exists, this is a reply; otherwise just “said”.
                            sb.AppendLine(seenAnyQuote ? $"{replyUsername} replied:" : $"{replyUsername} said:");
                            printedFirstNonQuote = true;
                        }
                        else
                        {
                            sb.AppendLine($"{replyUsername} said:");
                        }

                        sb.AppendLine(contrib.Text);
                        sb.AppendLine();
                    }
                }

                sb.AppendLine(new string('-', 40));
                sb.AppendLine();
            }

            return sb.ToString();
        }

        static void CreateOutputDocx(string filePath, List<ConversationBlock> blocks, string content)
        {
            if (File.Exists(filePath))
                File.Delete(filePath);

            using (WordprocessingDocument wordDoc =
                   WordprocessingDocument.Create(filePath, WordprocessingDocumentType.Document))
            {
                MainDocumentPart mainPart = wordDoc.AddMainDocumentPart();
                mainPart.Document = new Document(new Body());
                Body body = mainPart.Document.Body;

                string[] lines = content.Split(new[] { "\n" }, StringSplitOptions.None);

                int blockIndex = 0;

                foreach (var line in lines)
                {
                    Paragraph para = new Paragraph();

                    if (line.StartsWith("Topic Title:"))
                    {
                        string prefixTextStr = "Topic Title:  ";
                        int prefixLength = prefixTextStr.Length;
                        string shortTitle = line.Length >= prefixLength ? line.Substring(prefixLength).Trim() : "";

                        string linkUrl = (blockIndex < blocks.Count) ? blocks[blockIndex].TopicLink : "";

                        var prefixText = new Text(prefixTextStr) { Space = SpaceProcessingModeValues.Preserve };
                        Run prefixRun = new Run(prefixText);
                        para.Append(prefixRun);

                        if (!string.IsNullOrEmpty(linkUrl))
                        {
                            string relId = "rLink" + blockIndex;
                            mainPart.AddHyperlinkRelationship(new Uri(linkUrl), true, relId);

                            var docLink = new DocumentFormat.OpenXml.Wordprocessing.Hyperlink()
                            {
                                History = OnOffValue.FromBoolean(true),
                                Id = relId
                            };

                            Run linkRun = new Run(new Text(shortTitle));
                            linkRun.RunProperties = new RunProperties();
                            linkRun.RunProperties.Append(new Underline() { Val = UnderlineValues.Single });
                            linkRun.RunProperties.Append(new Color() { Val = "0000FF" });

                            docLink.Append(linkRun);
                            para.Append(docLink);
                        }
                        else
                        {
                            Run normalRun = new Run(new Text(shortTitle));
                            para.Append(normalRun);
                        }

                        blockIndex++;
                    }
                    else
                    {
                        Run normalRun = new Run(new Text(line));
                        para.Append(normalRun);
                    }

                    body.Append(para);
                }

                mainPart.Document.Save();
            }
        }
    }
}
